"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var react_component_ref_1 = require("@fluentui/react-component-ref");
var enzyme_1 = require("enzyme");
var React = require("react");
var react_test_renderer_1 = require("react-test-renderer");
var fixtures_1 = require("./fixtures");
var testInnerRef = function (Component) {
    var innerRef = jest.fn();
    var node = enzyme_1.mount(React.createElement(react_component_ref_1.RefFindNode, { innerRef: innerRef },
        React.createElement(Component, null))).getDOMNode();
    expect(innerRef).toHaveBeenCalledTimes(1);
    expect(innerRef).toHaveBeenCalledWith(node);
};
describe('RefFindNode', function () {
    describe('innerRef', function () {
        it('returns node from a functional component with DOM node', function () {
            testInnerRef(fixtures_1.DOMFunction);
        });
        it('returns node from a functional component', function () {
            testInnerRef(fixtures_1.CompositeFunction);
        });
        it('returns node from a class component with DOM node', function () {
            testInnerRef(fixtures_1.DOMClass);
        });
        it('returns node from a class component', function () {
            testInnerRef(fixtures_1.CompositeClass);
        });
        it('returns "null" after unmount', function () {
            var innerRef = jest.fn();
            var wrapper = enzyme_1.mount(React.createElement(react_component_ref_1.RefFindNode, { innerRef: innerRef },
                React.createElement(fixtures_1.CompositeClass, null)));
            innerRef.mockClear();
            wrapper.unmount();
            expect(innerRef).toHaveBeenCalledTimes(1);
            expect(innerRef).toHaveBeenCalledWith(null);
        });
        it('passes an updated node', function () {
            var innerRef = jest.fn();
            var wrapper = enzyme_1.mount(React.createElement(react_component_ref_1.RefFindNode, { innerRef: innerRef },
                React.createElement("div", null)));
            expect(innerRef).toHaveBeenCalledWith(expect.objectContaining({ tagName: 'DIV' }));
            wrapper.setProps({ children: React.createElement("button", null) });
            expect(innerRef).toHaveBeenCalledTimes(2);
            expect(innerRef).toHaveBeenCalledWith(expect.objectContaining({ tagName: 'BUTTON' }));
        });
        it('skips an update if node did not change', function () {
            var innerRef = jest.fn();
            var wrapper = enzyme_1.mount(React.createElement(react_component_ref_1.RefFindNode, { innerRef: innerRef },
                React.createElement("div", null)));
            expect(innerRef).toHaveBeenCalledWith(expect.objectContaining({ tagName: 'DIV' }));
            wrapper.setProps({ children: React.createElement("div", null) });
            expect(innerRef).toHaveBeenCalledTimes(1);
            expect(innerRef).toHaveBeenCalledWith(expect.objectContaining({ tagName: 'DIV' }));
        });
        it('handles updates of props', function () {
            var initialRef = jest.fn();
            var updatedRef = jest.fn();
            var wrapper = enzyme_1.mount(React.createElement(react_component_ref_1.RefFindNode, { innerRef: initialRef },
                React.createElement("div", null)));
            expect(initialRef).toHaveBeenCalled();
            expect(updatedRef).not.toHaveBeenCalled();
            jest.resetAllMocks();
            wrapper.setProps({ innerRef: updatedRef });
            expect(initialRef).not.toHaveBeenCalled();
            expect(updatedRef).toHaveBeenCalled();
        });
        it('always returns "null" for react-test-renderer', function () {
            var innerRef = jest.fn();
            var ref = jest.fn();
            react_test_renderer_1.create(React.createElement(react_component_ref_1.RefFindNode, { innerRef: innerRef },
                React.createElement("div", { ref: ref })));
            expect(innerRef).toHaveBeenCalledWith(null);
            expect(ref).toHaveBeenCalledWith(null);
        });
    });
});
